#!/usr/bin/env python3
# Copyright (c) Facebook, Inc. and its affiliates. All Rights Reserved.

"""Wrapper to train and test a video classification model."""
from slowfast.config.defaults import assert_and_infer_cfg
from slowfast.utils.misc import launch_job
from slowfast.utils.parser import load_config, parse_args


def main():
    """
    Main function to spawn the train and test process.
    """
    args = parse_args()
    cfg = load_config(args)
    cfg = assert_and_infer_cfg(cfg)
    if cfg.JOINT_LEARNING.DN_TYPE == "nn":
        # Importing functions for the DN-NN model
        from dn_nn.test_net import test
        from dn_nn.train_net import train
    elif cfg.JOINT_LEARNING.DN_TYPE == "lr":
        # Importing functions for the DN-LR model
        from dn_lr.test_net import test
        from dn_lr.train_net import train
    else:
        raise TypeError(
                "Incorrect Dependency Network Type Given, Please select one from Neural Network model(nn) or Logistic "
                "regression model(lr) for the key JOINT_LEARNING.DN_TYPE")
    # Perform training.
    if cfg.TRAIN.ENABLE:
        launch_job(cfg=cfg, init_method=args.init_method, func=train)

    print(cfg.JOINT_LEARNING.MODEL_DIRECTORY)
    # Perform multi-clip testing.
    if cfg.TEST.ENABLE:
        launch_job(cfg=cfg, init_method=args.init_method, func=test)


if __name__ == "__main__":
    main()
